package matrices

/**
 * A simple class for square matrices with Double entries.
 */
class Matrix(val n: Int) {
  require(n > 0, "The dimension n must be positive")
  protected[Matrix] val entries = new Array[Double](n * n)

  /* With this we can access elements by writing M(i,j) */
  def apply(row: Int, column: Int) = {
    require(0 <= row && row < n)
    require(0 <= column && column < n)
    entries(row * n + column)
  }

  /* With this we can set elements by writing M(i,j) = v */
  def update(row: Int, column: Int, value: Double) : Unit = {
    require(0 <= row && row < n)
    require(0 <= column && column < n)
    entries(row * n + column) = value
  }

  /* Returns a new matrix that is the sum of this and that */
  def +(that: Matrix): Matrix = {
    val result = new Matrix(n)
    for (row <- 0 until n; column <- 0 until n)
      result(row, column) = this(row, column) + that(row, column)
    result
  }

  /* Returns a new matrix that is the product of this and that */
  def *(that: Matrix): Matrix = {
    val result = new Matrix(n)
    for (row <- 0 until n; column <- 0 until n) {
      var v = 0.0
      for (i <- 0 until n)
        v += this(row, i) * that(i, column)
      result(row, column) = v
    }
    result
  }
}
